/*
 * Decompiled with CFR 0.152.
 */
package gov.noaa.tsunami.websift.propdb;

import gov.noaa.tsunami.websift.propdb.DirectoryMonitor;
import gov.noaa.tsunami.websift.propdb.SourceCombo;
import gov.noaa.tsunami.websift.propdb.UnitSource;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

public final class PropagationDatabase
implements ActionListener {
    private static final PropagationDatabase _instance = new PropagationDatabase();
    private final Map<String, UnitSource> unitSources = new LinkedHashMap<String, UnitSource>();
    private final Map<String, DirectoryMonitor> dirMonitors = new HashMap<String, DirectoryMonitor>();
    public static Logger propdbLogger = Logger.getLogger("gov.noaa.tsunami");
    private File infoSzFile;
    private static final double Re = 6371.22;
    private static final double D2R = Math.PI / 180;

    private PropagationDatabase() {
    }

    public static PropagationDatabase getInstance() {
        return _instance;
    }

    public void setSourceDirs(String[] sourceDirs) {
        if (sourceDirs != null) {
            this.scanPropFiles(sourceDirs);
        }
    }

    public Logger getLogger() {
        return propdbLogger;
    }

    public void setLogger(Logger l) {
        propdbLogger = l;
    }

    public final void scanPropFiles(String[] sourceDirs) {
        int changesMade = 0;
        boolean updateNow = this.infoSzFile != null && !this.infoSzFile.exists();
        for (String spath : sourceDirs) {
            if (!updateNow && this.dirMonitors.containsKey(spath)) continue;
            File dir = new File(spath);
            if (dir.exists()) {
                FilenameFilter haFilter = new FilenameFilter(){

                    @Override
                    public boolean accept(File dir, String name) {
                        return name.endsWith("ha.nc");
                    }
                };
                for (File file : dir.listFiles(haFilter)) {
                    try {
                        UnitSource newSource = new UnitSource(file);
                        if (!this.addOrCombineSource(newSource)) continue;
                        ++changesMade;
                    }
                    catch (IOException ex) {
                        propdbLogger.log(Level.WARNING, "IO exception reading source file {0}: {1}", new Object[]{file.getPath(), ex.getMessage()});
                    }
                }
                this.dirMonitors.put(spath, new DirectoryMonitor(new File(spath), this));
                continue;
            }
            propdbLogger.log(Level.WARNING, "source directory doesn''t exist: {0}", spath);
        }
        if (changesMade > 0 && this.infoSzFile != null) {
            propdbLogger.log(Level.INFO, changesMade + " propagation database files changed; updating index file.");
            this.updateInfoSzFile();
        }
    }

    public void setIndexFile(File infosz) {
        this.infoSzFile = infosz;
        if (!this.unitSources.isEmpty()) {
            this.updateInfoSzFile();
        }
    }

    public File getIndexFile() {
        return this.infoSzFile;
    }

    private void updateInfoSzFile() {
        if (this.infoSzFile == null) {
            return;
        }
        String timestamp = new SimpleDateFormat().format(Calendar.getInstance().getTime());
        try {
            PrintWriter writer = new PrintWriter(this.infoSzFile);
            writer.println("# Propagation database metadata file");
            writer.println("# Generated from source files: " + timestamp);
            writer.print("# Source database directories: ");
            for (String dir : this.dirMonitors.keySet()) {
                writer.print(" " + dir);
            }
            writer.println("\n# Name, Filename/URL, Long(deg), Lat(deg), Slip(m), Strike(deg), Dip(deg), Depth(km), Length(km), Width(km), Rake(deg)");
            for (UnitSource us : this.unitSources.values()) {
                writer.printf("%s, %s, %.9f, %.9f, %f, %.9f, %f, %f, %f, %f, %f\n", us.getName(), us.getFileName() != null ? us.getFileName() : us.getCompressedFileName(), us.getLongitude(), us.getLatitude(), us.getSlip(), us.getStrike(), us.getDip(), us.getDepth(), us.getLength(), us.getWidth(), us.getRake());
            }
            writer.close();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void actionPerformed(ActionEvent evt) {
        String command = evt.getActionCommand();
        String[] sa = command.split(",");
        String fileName = sa[1] + File.separator + sa[2];
        propdbLogger.log(Level.INFO, "Source file change: {0}", command);
        if (command.startsWith("added") || command.startsWith("changed") && fileName.endsWith("ha.nc")) {
            try {
                UnitSource aSource = new UnitSource(new File(fileName));
                this.addOrCombineSource(aSource);
                this.updateInfoSzFile();
            }
            catch (IOException ex) {
                propdbLogger.log(Level.WARNING, "Exception reading new source", ex);
            }
        } else if (command.startsWith("deleted") && fileName.endsWith("ha.nc")) {
            Iterator<UnitSource> iter = this.unitSources.values().iterator();
            while (iter.hasNext()) {
                UnitSource aSource = iter.next();
                if (aSource.getCompressedFileName().equals(fileName)) {
                    aSource.setCompressedFileName(null);
                }
                if (aSource.getFileName().equals(fileName)) {
                    aSource.setFileName(null);
                }
                if (aSource.getFileName() != null || aSource.getCompressedFileName() != null) continue;
                iter.remove();
            }
        } else if (command.startsWith("noexist")) {
            String dir = sa[1];
            Iterator<UnitSource> iter = this.unitSources.values().iterator();
            while (iter.hasNext()) {
                UnitSource aSource = iter.next();
                if (aSource.getFileName().indexOf(dir) != -1) {
                    aSource.setFileName(null);
                }
                if (aSource.getCompressedFileName().indexOf(dir) != -1) {
                    aSource.setCompressedFileName(null);
                }
                if (aSource.getFileName() != null || aSource.getCompressedFileName() != null) continue;
                iter.remove();
            }
            this.dirMonitors.remove(dir);
        }
    }

    private boolean addOrCombineSource(UnitSource newSource) {
        UnitSource existing = this.unitSources.get(newSource.getName());
        boolean changed = false;
        if (existing != null) {
            if (newSource.getFileName() != null && existing.getFileName() == null) {
                existing.setFileName(newSource.getFileName());
                changed = true;
            } else if (newSource.getCompressedFileName() != null && existing.getCompressedFileName() == null) {
                existing.setCompressedFileName(newSource.getCompressedFileName());
                changed = true;
            }
        } else {
            this.unitSources.put(newSource.getName(), newSource);
            changed = true;
        }
        return changed;
    }

    public Map<String, UnitSource> getUnitSourceMap() {
        return this.unitSources;
    }

    public ArrayList<UnitSource> getUnitSources() {
        return new ArrayList<UnitSource>(this.unitSources.values());
    }

    public UnitSource getUnitSource(String sourceName) {
        return this.unitSources.get(sourceName);
    }

    public SourceCombo getSourceCombo(String linearCombo) {
        SourceCombo sc = new SourceCombo();
        if (linearCombo == null) {
            return sc;
        }
        String[] sa = linearCombo.split("(?<!\\*)\\+|(?<!\\*)(?=-)");
        for (int i = 0; i < sa.length; ++i) {
            UnitSource us;
            double alpha;
            String[] slipAndSource = sa[i].split("\\*");
            if (slipAndSource.length != 2) continue;
            try {
                alpha = Double.parseDouble(slipAndSource[0].trim());
                us = this.getUnitSource(slipAndSource[1].trim());
            }
            catch (NumberFormatException nfe) {
                alpha = Double.parseDouble(slipAndSource[1].trim());
                us = this.getUnitSource(slipAndSource[0].trim());
            }
            if (us == null) continue;
            us = us.copy();
            us.setAlpha(alpha);
            sc.addSource(us);
        }
        return sc;
    }

    public void setUnitSources(Collection<UnitSource> sources) {
        this.unitSources.clear();
        for (UnitSource aSource : sources) {
            this.unitSources.put(aSource.getName(), aSource);
        }
    }

    public void clearUnitSources() {
        this.unitSources.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void useCroppedSources(InputStream is) throws IOException {
        propdbLogger.entering("PropagationDatabase", "useCroppedSources");
        ArrayList<UnitSource> csv = this.readDatabaseInfo(is, "", false);
        Map<String, UnitSource> map = this.unitSources;
        synchronized (map) {
            for (UnitSource us : csv) {
                UnitSource existing = this.getUnitSource(us.getName());
                if (!us.isCropped()) {
                    propdbLogger.log(Level.WARNING, "Unit source {0} not marked as cropped in local database", us);
                }
                if (existing == null) {
                    propdbLogger.log(Level.FINEST, "extra cropped source: {0}", us.getName());
                    this.unitSources.put(us.getName(), us);
                    continue;
                }
                propdbLogger.log(Level.FINER, "cropped source: {0}", us.getName());
                existing.setCroppedFileName(us.getFileName());
            }
        }
        propdbLogger.exiting("PropagationDatabase", "useCroppedSources");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeCroppedSources() {
        propdbLogger.entering("PropagationDatabase", "removeCroppedSources");
        Map<String, UnitSource> map = this.unitSources;
        synchronized (map) {
            Iterator<UnitSource> bi = this.unitSources.values().iterator();
            while (bi.hasNext()) {
                UnitSource src = bi.next();
                src.setCroppedFileName(null);
                if (src.getFileName() != null) continue;
                bi.remove();
            }
        }
        propdbLogger.exiting("PropagationDatabase", "removeCroppedSources");
    }

    public void setDatabaseLocations(List<String> v, boolean global) throws IOException {
        this.setDatabaseLocations(v, null, global);
    }

    public void setDatabaseLocations(List<String> dblocs, String proxy, boolean global) throws IOException {
        this.unitSources.clear();
        for (String urlName : dblocs) {
            this.addDatabaseLocation(urlName, proxy, global);
        }
        Collections.sort(new ArrayList<UnitSource>(this.unitSources.values()));
    }

    public void addDatabaseLocation(String location) throws IOException {
        this.addDatabaseLocation(location, null, null, false);
    }

    public void addDatabaseLocation(String location, String proxy, boolean global) throws IOException {
        this.addDatabaseLocation(location, proxy, null, global);
    }

    public void addDatabaseLocation(String location, boolean global) throws IOException {
        this.addDatabaseLocation(location, null, null, global);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addDatabaseLocation(String location, String proxy, String reqValue, boolean global) throws IOException {
        propdbLogger.log(Level.INFO, "adding sources from prop database: {0} proxy: {1}", new Object[]{location, proxy});
        ArrayList<UnitSource> newSources = Collections.EMPTY_LIST;
        if (location.startsWith("http")) {
            String qparams = global ? "?global=true" : "";
            URL url = new URL(location + "/info_sz.dat" + qparams);
            propdbLogger.info("url: " + url);
            HttpURLConnection conn = null;
            try (InputStream inStream = null;){
                conn = (HttpURLConnection)url.openConnection();
                if (proxy != null) {
                    if (reqValue != null) {
                        conn.setRequestProperty(proxy, reqValue);
                    } else {
                        conn.setRequestProperty("Proxy-Authorization", proxy);
                    }
                }
                conn.setConnectTimeout(60000);
                conn.setReadTimeout(60000);
                inStream = conn.getInputStream();
                String encoding = conn.getContentEncoding();
                if (encoding != null && encoding.equalsIgnoreCase("gzip")) {
                    propdbLogger.info("reading info_sz.dat with gzip");
                    inStream = new GZIPInputStream(inStream);
                } else if (encoding != null && encoding.equalsIgnoreCase("deflate")) {
                    propdbLogger.info("reading info_sz.dat with deflate");
                    inStream = new InflaterInputStream(inStream, new Inflater(true));
                } else {
                    propdbLogger.info("reading info_sz.dat uncompressed");
                }
                newSources = this.readDatabaseInfo(inStream, "", false);
            }
        } else {
            File locdir;
            File localf = new File(location + File.separator + "info_sz.dat");
            if (localf.canRead()) {
                newSources = this.readDatabaseInfo(new FileInputStream(localf), "", false);
            }
            if (newSources.isEmpty() && (locdir = new File(location)).canWrite()) {
                this.infoSzFile = localf;
                this.scanPropFiles(new String[]{location});
                this.updateInfoSzFile();
            }
        }
        if (newSources.size() > 0) {
            Map<String, UnitSource> map = this.unitSources;
            synchronized (map) {
                Iterator it = newSources.iterator();
                while (it.hasNext()) {
                    UnitSource addSource = (UnitSource)it.next();
                    if (!this.unitSources.containsKey(addSource.getName())) continue;
                    propdbLogger.log(Level.INFO, "duplicate found: {0} (using old UnitSource)", addSource.getName());
                    it.remove();
                }
                for (UnitSource aSource : newSources) {
                    this.unitSources.put(aSource.getName(), aSource);
                }
            }
        }
        propdbLogger.log(Level.INFO, "unit sources available: {0}", this.unitSources.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addUnitSource(UnitSource aSource) {
        Map<String, UnitSource> map = this.unitSources;
        synchronized (map) {
            if (this.unitSources.containsKey(aSource.getName())) {
                propdbLogger.log(Level.INFO, "dpublicate found: {0} (using old UnitSource)", aSource.getName());
            } else {
                this.unitSources.put(aSource.getName(), aSource);
            }
        }
    }

    private ArrayList<UnitSource> readDatabaseInfo(InputStream is, String path, boolean checkSource) throws IOException {
        String inputLine;
        ArrayList<UnitSource> result = new ArrayList<UnitSource>();
        String sn = "";
        String fn = "";
        String zone = "";
        double lon = 0.0;
        double lat = 0.0;
        double slip = 0.0;
        double strike = 0.0;
        double dip = 0.0;
        double depth = 0.0;
        double length = 0.0;
        double width = 0.0;
        double rake = 0.0;
        boolean cropped = false;
        BufferedReader br = new BufferedReader(new InputStreamReader(is));
        while ((inputLine = br.readLine()) != null) {
            File testFile;
            String[] lineArray;
            if (Thread.interrupted()) {
                propdbLogger.fine("thread interrupted while reading database; no data returned");
                result.clear();
                result.trimToSize();
                return result;
            }
            if (inputLine.startsWith("#") || (lineArray = inputLine.split(",")).length != 10 && lineArray.length != 7 && lineArray.length != 11 && lineArray.length != 12) continue;
            cropped = false;
            slip = 1.0;
            length = 100.0;
            width = 50.0;
            rake = 90.0;
            int count = 0;
            sn = lineArray[count].trim();
            fn = lineArray[++count].trim();
            if ((lon = Double.parseDouble(lineArray[++count].trim())) < 0.0) {
                lon += 360.0;
            }
            lat = Double.parseDouble(lineArray[++count].trim());
            ++count;
            if (lineArray.length == 11 || lineArray.length == 12) {
                slip = Double.parseDouble(lineArray[count].trim());
                ++count;
            }
            strike = Double.parseDouble(lineArray[count].trim());
            dip = Double.parseDouble(lineArray[++count].trim());
            depth = Double.parseDouble(lineArray[++count].trim());
            ++count;
            if (lineArray.length == 10 || lineArray.length == 11 || lineArray.length == 12) {
                length = Double.parseDouble(lineArray[count].trim());
                width = Double.parseDouble(lineArray[++count].trim());
                rake = Double.parseDouble(lineArray[++count].trim());
                ++count;
            }
            if (lineArray.length == 12) {
                cropped = lineArray[count].trim().equals("c");
                ++count;
            }
            if (checkSource && !(testFile = new File(path + File.separator + fn)).exists()) {
                propdbLogger.log(Level.INFO, "Propagation file not found, and checkSource required, not adding source: {0}", sn);
                continue;
            }
            if (path.length() != 0) {
                fn = path + File.separator + fn;
            }
            UnitSource aSource = new UnitSource(sn, cropped ? null : fn, lon, lat, slip, strike, dip, depth, length, width, rake);
            if (cropped) {
                aSource.setCroppedFileName(fn);
            }
            result.add(aSource);
        }
        is.close();
        return result;
    }

    public SourceCombo getSeismicInversion(double magnitude, double longitude, double latitude) {
        SourceCombo sc = new SourceCombo();
        ArrayList<UnitSource> sources = new ArrayList<UnitSource>();
        sources.addAll(this.unitSources.values());
        if (sources.size() < 3) {
            return sc;
        }
        PropagationDatabase.sortSourcesByDistance(sources, longitude, latitude);
        UnitSource closest = sources.get(0);
        if (closest == null) {
            return sc;
        }
        int snum = closest.getNameNumber();
        String slet = closest.getNameLetter();
        boolean outerRise = false;
        if (slet.equals("c") || slet.equals("d")) {
            outerRise = true;
        }
        int nrows = 1;
        int ncols = 1;
        if (magnitude <= 8.1) {
            sc.addSource(closest);
            sc.setMomentMagnitude(magnitude);
            return sc;
        }
        if (magnitude > 8.1 && magnitude <= 8.3) {
            nrows = 2;
            ncols = 1;
        } else if (magnitude > 8.3 && magnitude <= 8.4) {
            nrows = 3;
            ncols = 1;
        } else if (magnitude > 8.4 && magnitude <= 8.6) {
            nrows = 3;
            ncols = 2;
        } else if (magnitude > 8.6 && magnitude <= 8.8) {
            nrows = 4;
            ncols = 2;
        } else if (magnitude > 8.8 && magnitude <= 9.1) {
            nrows = 5;
            ncols = 3;
        } else if (magnitude > 9.1 && magnitude <= 9.2) {
            nrows = 6;
            ncols = 3;
        } else if (magnitude > 9.2 && magnitude <= 9.3) {
            nrows = 7;
            ncols = 3;
        } else if (magnitude > 9.3) {
            nrows = 8;
            ncols = 3;
        }
        HashSet<Integer> rows = new HashSet<Integer>();
        int count = 1;
        UnitSource source = sources.get(count);
        while (rows.size() < nrows && count < 30) {
            rows.add(source.getNameNumber());
            if (sources.size() == count) {
                return sc;
            }
            source = sources.get(count++);
        }
        for (Integer r : rows) {
            String let2 = "z";
            String let = "y";
            if (outerRise) {
                source = this.getUnitSource(String.format("%s%d%s", closest.getZone(), r, slet));
                if (source != null) {
                    sc.addSource(source);
                }
                if (ncols <= 1 || (source = slet.equals("c") ? this.getUnitSource(String.format("%s%dd", closest.getZone(), r)) : this.getUnitSource(String.format("%s%dd", closest.getZone(), r))) == null) continue;
                sc.addSource(source);
                continue;
            }
            source = this.getUnitSource(String.format("%s%d%s", closest.getZone(), r, slet));
            if (source != null) {
                sc.addSource(source);
            }
            if (ncols == 2) {
                switch (slet) {
                    case "b": {
                        let = "a";
                        break;
                    }
                    case "a": {
                        let = "b";
                        break;
                    }
                    case "z": {
                        let = "a";
                    }
                }
                source = this.getUnitSource(String.format("%s%d%s", closest.getZone(), r, let));
                if (source != null) {
                    sc.addSource(source);
                }
            }
            if (ncols != 3) continue;
            switch (slet) {
                case "b": {
                    let = "a";
                    let2 = "z";
                    break;
                }
                case "a": {
                    let = "b";
                    let2 = "z";
                    break;
                }
                case "z": {
                    let = "b";
                    let2 = "a";
                    break;
                }
                case "y": {
                    let = "a";
                    let2 = "z";
                }
            }
            source = this.getUnitSource(String.format("%s%d%s", closest.getZone(), r, let));
            if (source != null) {
                sc.addSource(source);
            }
            if ((source = this.getUnitSource(String.format("%s%d%s", closest.getZone(), r, let2))) == null) continue;
            sc.addSource(source);
        }
        sc.setMomentMagnitude(magnitude);
        return sc;
    }

    public SourceCombo getSeismicInversion_orig(double magnitude, double longitude, double latitude) {
        SourceCombo sc = new SourceCombo();
        int numberOfSources = 1;
        if (magnitude > 7.8 && magnitude <= 8.1) {
            numberOfSources = 2;
        } else if (magnitude > 8.1 && magnitude <= 8.3) {
            numberOfSources = 4;
        } else if (magnitude > 8.3 && magnitude <= 8.5) {
            numberOfSources = 6;
        } else if (magnitude > 8.5) {
            numberOfSources = 8;
        }
        if (this.unitSources.size() < numberOfSources) {
            return sc;
        }
        ArrayList<UnitSource> sources = new ArrayList<UnitSource>();
        for (UnitSource u : this.unitSources.values()) {
            String c = u.getNameLetter();
            if (!c.equals("a") && !c.equals("b") && !c.equals("z")) continue;
            sources.add(u);
        }
        PropagationDatabase.sortSourcesByDistance(sources, longitude, latitude);
        for (int i = 0; i < numberOfSources; ++i) {
            sc.addSource(sources.get(i).copy());
        }
        sc.setMomentMagnitude(magnitude);
        return sc;
    }

    public UnitSource findPair(UnitSource us) {
        UnitSource pairSource2 = null;
        for (UnitSource pairSource2 : this.unitSources.values()) {
            if (pairSource2.getZone().equals(us.getZone()) && pairSource2.getNameNumber() == us.getNameNumber() && !pairSource2.getNameLetter().equals(us.getNameLetter())) break;
        }
        return pairSource2;
    }

    public static double distanceBetween(double lat1, double lon1, double lat2, double lon2) {
        double theta = Math.sin(lat1 * (Math.PI / 180)) * Math.sin(lat2 * (Math.PI / 180)) + Math.cos(lat1 * (Math.PI / 180)) * Math.cos(lat2 * (Math.PI / 180)) * Math.cos((lon1 - lon2) * (Math.PI / 180));
        theta = Math.acos(theta);
        return 6371.22 * theta;
    }

    public static void sortSourcesByDistance(List<UnitSource> sources, double lon, double lat) {
        for (UnitSource aSource : sources) {
            aSource.setDistanceFromSource(PropagationDatabase.distanceBetween(aSource.getLatitude(), aSource.getLongitude(), lat, lon));
        }
        Collections.sort(sources, new distComparator());
    }

    private static class distComparator
    implements Comparator<UnitSource> {
        private distComparator() {
        }

        @Override
        public int compare(UnitSource o1, UnitSource o2) {
            int i = 0;
            if (o1.getDistanceFromSource() < o2.getDistanceFromSource()) {
                i = -1;
            } else if (o1.getDistanceFromSource() > o2.getDistanceFromSource()) {
                i = 1;
            }
            return i;
        }
    }

    public static enum ModelVariable {
        HA,
        UA,
        VA;

    }
}

